-- Fichero FSQL_PKG.sql
----------------------------------------------------------------------
-- * PAQUETE PL/SQL: FSQL_PKG
-- * OBJETIVO      : Servidor FSQL
-- * AUTOR         : Jos Galindo G.
-- * DIRECCIN e idea original: J.M. Medina R.
-- * FECHAS: Entre Junio 1997 y Diciembre 1997 (Vaya verano!).

-- * MODO de EMPLEO:
--   La funcin FSQL2SQL traduce una consulta en FSQL a su equivalente en SQL.
--   Devuelve el nmero de errores cometidos (0 si no hubo errores).
--   Si el primer carcter es '!' no efecta la traduccin.
--   # En caso de 0 errores, el resultado, la consulta en SQL, estar almacenada en
--     la tabla FSQL_QUERY y ser obtenido concatenando el campo FSQL_QUERY.ATRIBUTO
--     de las tuplas de la sesin actual (FSQL_QUERY.SESSIONID=USERENV('SESSIONID')),
--     ordenados por el campo 'INDICE'.
--     -> NOTA: En dicha concatenacin habr que dejar un espacio entre cada valor de
--         'ATRIBUTO'. Este espacio se puede omitir si 'ATRIBUTO' tiene longitud 1 y
--         su primer carcter vale uno de los siguientes caracteres:
--         '+', '-', '*', '/', '.', ';', ',', '=', '>', '<', '(' y ')'.
--   # En caso de encontrar errores, los mensajes de error correspondientes se encuentran
--     en el campo FSQL_ERRORS.MSG_ERROR de las tuplas de la sesin actual (SESSIONID),
--     ordenados en orden de aparicin por el campo INDICE.

-- * La consulta puede incluir comentarios:
--   # Entre /* y */                      comentarios como en C
--   # Con una barra y un asterisco: /*   Hasta final de la consulta
--   # Con dos guiones --                 Hasta final de lnea

-- * REQUISITOS de FUNCIONAMIENTO de este PAQUETE:
--   Adems de las tablas propias de FIRST (FUZZY_COL_LIST...), este paquete necesita
--   tener creadas las tablas que requieren los analizadores (Lxico, Sintctico y Semntico):
--	----------------------------------------------------------------------------------------------
--	TABLA			LEIDA POR		ESCRITA POR		OWNER	Contenido
--	----------------------------------------------------------------------------------------------
--	T_TRANSI		A. Lxico		CONSTANTE		SYS	Transiciones del Autmata
--	RESERVADAS		A. Lxico		CONSTANTE		SYS	Palabras Reservadas
--	TABLA_SINTAX	A. Sintctico	CONSTANTE		SYS	Transiciones de la Gramtica
--	PRODUCCIONES	A. Sintctico	CONSTANTE		SYS	Producciones de la Gramtica
--	FSQL_ALL_ERRORS	--			--			SYS	Mensajes de los Errores
--	FSQL_ALL_QUERIES	--			--			SYS	Consultas: FSQL y su traduccin SQL
--	Vista FSQL_ERRORS	Cliente FSQL	Lx/Sint/Sem	SYS	Vista sobre FSQL_ALL_ERRORS particular
--	Vista FSQL_QUERY	Sint/Sem/Cliente	Lx/Sint/Sem	SYS	Vista sobre FSQL_ALL_QUERIES particular
--	----------------------------------------------------------------------------------------------
--    # Observaciones:
--		1. Las tablas/vistas deben ser propiedad del administrador (SYS).
--		2. Las vistas que son LEIDAS por el Cliente FSQL (FSQL_*) son las nicas que leer el Cliente
--		   y ambas estn personalizadas para cada usuario: Cada usuario slo puede acceder,
--		   a travs de estas vistas, a las tuplas de las tablas FSQL_ALL_* que tienen su SESSIONID.
--		   Estas vistas deben tener permiso de lectura/escritura todos los usuarios que puedan usar FSQL
--		   (normalmente sern vistas pblicas).
--		3. Las otras 4 tablas son privadas y sus datos se introducirn en la instalacin y permanecern CONSTANTES
--		   (mientras no se modifique la versin del servidor FSQL).
--		4. El 'Cliente FSQL' ser un programa que efecte consultas en FSQL. Su modo de actuar es:
--			a) Formar la consulta FSQL
--			b) Ejecutar: Numero_Errores = FSQL2SQL (consulta_FSQL)
--			c) IF (Numero_Errores > 0) THEN
--			      Mostrar errores (Accediendo a FSQL_ERRORS ordenados por INDICE)
--		         ELSE
--			      Formar la consulta SQL (Accediendo a FSQL_QUERY y ordenados por INDICE)
--			      Enviar consulta a Oracle
--			   END IF;
--			d) Al final, el cliente debe borrar las tuplas insertadas en las tablas FSQL_* (tuplas con su SESSIONID):
--			   Ejecutar procedimiento FSQL_FIN (de FSQL_PKG).

-- * REQUISITOS de FUNCIONAMIENTO de la SENTENCIA SQL resultante:
--   Para que la sentencia SQL resultante de FSQL2SQL funcione correctamente
--   se requiere el paquete de funciones FSQL_FUNCTIONS que implementa las
--   funciones que incorpora la traduccin a SQL.
----------------------------------------------------------------------
CREATE OR REPLACE PACKAGE FSQL_PKG AS

  -- Funcin de traduccin de FSQL a SQL: Devuelve el nmero de errores
  FUNCTION  FSQL2SQL (consulta IN LONG) RETURN INTEGER;

  -- Funcin para concatenar la sentencia SQL resultante, devolvindola en una cadena.
  -- Puede usarse tras FSQL2SQL, si no hubo errores.
  FUNCTION FSQL_concat RETURN VARCHAR2;

  -- Procedimiento para terminar una sesin usando FSQL: Borra datos temporales
  PROCEDURE FSQL_FIN;

  -- Devuelve el nmero de OBJeto de una tabla: sch.tab
  FUNCTION FSQL_OBJ (sch IN VARCHAR2, tab IN VARCHAR2)
    RETURN ACCESSIBLE_TABLES.OBJ#%TYPE;

  -- Devuelve el nmero de identificador de una COLumna: sch.tab.col
  FUNCTION FSQL_COL (sch IN VARCHAR2, tab IN VARCHAR2, col IN VARCHAR2)
    RETURN DBA_TAB_COLUMNS.COLUMN_ID%TYPE;

  -- Devuelve tipo de atributo difuso de una columna: OBJ_ID/COL_ID
  FUNCTION FSQL_FTYPE (OBJ_ID IN ACCESSIBLE_TABLES.OBJ#%TYPE, COL_ID IN DBA_TAB_COLUMNS.COLUMN_ID%TYPE)
    RETURN DBA_TAB_COLUMNS.COLUMN_ID%TYPE;

  -- Por si quiere usarla un Cliente FSQL para insertar sus propios errores
  -- en la tabla de errores FSQL_ERRORS
  PROCEDURE insertar_error (MsgError IN VARCHAR2);
END FSQL_PKG;
/

CREATE OR REPLACE PACKAGE BODY FSQL_PKG AS
  consulta  CONSTANT INTEGER:= 301;
  S_INICIAL CONSTANT PRODUCCIONES.parte_der%TYPE := 'consulta';
  FIN       CONSTANT PRODUCCIONES.parte_der%TYPE := 'THE END';
  AUDSID    FSQL_QUERY.SESSIONID%TYPE; -- Identificativo de la sesin actual

  -- Tipos para la PILA del A. sintctico
  TYPE TPILA_SIMBOLO  IS TABLE OF PRODUCCIONES.parte_der%TYPE
                         INDEX BY BINARY_INTEGER;
  TYPE TPILA_TERMINAL IS TABLE OF PRODUCCIONES.terminal_der%TYPE
                         INDEX BY BINARY_INTEGER;
  TYPE TPILA IS RECORD
       (tope     NUMBER(5),
        simbolo  TPILA_SIMBOLO,
        terminal TPILA_TERMINAL);

----------------------------------------------------------------------
-- Inserta un msg de error en la tabla de errores: FSQL_ERRORS
-- La insercin ser realiza cronolgicamente ordenados por el campo INDICE (empezando por 1)
-- Se inserta el nmero de la sesin actual en el campo SESSIONID.
----------------------------------------------------------------------
PROCEDURE insertar_error (MsgError IN VARCHAR2) IS
BEGIN
  FSQL_AUX.inserta_error(MsgError);
END insertar_error;


-------------------------------------------------------------------------------------
-- ***** ***** ***** ***** *****  ANALIZADOR LXICO  ***** ***** ***** ***** ***** --
-------------------------------------------------------------------------------------

----------------------------------------------------------------------
-- Inserta un token en la tabla
-- FSQL_QUERY: indice es un nmero incrementado sucesivamente (desde 0)
--              nombre es el nombre del token
--              atributo es la cadena que forma el token (sin espacios)
-- Si el estado en el que queda el autmata tras ese token es un estado NO terminal
-- devuelve un cdigo de error. Devuelve 0 si es un estado Terminal y el token se
-- insert en la tabla correctamente.
----------------------------------------------------------------------
FUNCTION inserta_token (cadena   IN VARCHAR2,
                        estado   IN INTEGER,
                        posicion IN FSQL_QUERY.posicion%TYPE) RETURN INTEGER IS
  i      INTEGER;
  indi   FSQL_QUERY.indice%TYPE;
  nomtok FSQL_QUERY.nombre%TYPE;
BEGIN
    IF    (estado=18) THEN
      FSQL_AUX.inserta_error('ERROR LXICO en posicin '||posicion||': '||
        'Comparador de desigualdad (!=,^=) no terminado.');
      RETURN 2; -- ERROR: Comparador de desigualdad (!=  ^=) no terminado
    ELSIF (estado=23) OR (estado=25) THEN
      FSQL_AUX.inserta_error('ERROR LXICO en posicin '||posicion||': '||
        'Cadena o texto no terminado correctamente (faltan comillas).');
      RETURN 3; -- ERROR: CADENA o TEXTO no terminado (falta "  ')
    ELSIF (estado=31) OR (estado=32) THEN
      FSQL_AUX.inserta_error('ERROR LXICO en posicin '||posicion||': '||
        'Nmero en notacin exponencial mal terminado.');
      RETURN 4; -- ERROR: Nmero exponencial mal terminado
    ELSIF (estado=35) THEN
      FSQL_AUX.inserta_error('ERROR LXICO en posicin '||posicion||': '||
        'Operador de concatenacin || mal terminado.');
      RETURN 5; -- ERROR: Operador || incompleto
    ELSIF (estado = 9)  THEN nomtok := 'GT';
    ELSIF (estado = 11) THEN nomtok := 'LT';
    ELSIF (estado = 12) THEN nomtok := 'EQ';
    ELSIF (estado = 16) THEN nomtok := 'GEQ';
    ELSIF (estado = 17) THEN nomtok := 'LEQ';
    ELSIF (estado = 19) THEN nomtok := 'NEQ';
    ELSIF (estado = 24) THEN nomtok := 'CADENA';
    ELSIF (estado = 26) THEN nomtok := 'TEXTO';
    ELSIF (estado = 13) OR   -- Nmero entero
          (estado = 14) OR   -- Nmero con decimales
          (estado = 33) THEN -- Nmero en notacin exponencial
      nomtok := 'NUMERO';
    ELSIF (estado = 15) THEN -- Ver si es un Identificador o una Plabra. Reservada
      SELECT count(*) INTO i FROM RESERVADAS WHERE palabra=cadena;
      IF i=0 THEN
        nomtok := 'ID';
      ELSE -- i=1
        nomtok := cadena;
      END IF;
    ELSIF estado IN (37,38,39,40) THEN -- Si es un comentario, no se inserta nada
      RETURN 0; -- 38 es estado terminal y 37,39,40 por si termina la consulta sin llegar al 38
    ELSE
      nomtok := cadena; -- Otros smbolos: puntos, comas, parntesis...
    END IF;
    SELECT MAX(indice) into indi from FSQL_QUERY;
    IF indi >= 0 THEN
         indi:=indi+1;
    ELSE indi:=0;
    END IF;
--    dbms_output.put_line(indi||'--'||nomtok||'--'||cadena);
    INSERT INTO FSQL_QUERY VALUES(AUDSID,indi,posicion,nomtok,cadena);
    RETURN 0;
END inserta_token;

----------------------------------------------------------------------
-- ANALIZADOR LXICO para una consulta en FSQL
-- Usa     : Tabla de palabras reservadas (RESERVADAS)
--           Tabla de transiciones del autmata (T_TRANSI)
-- Genera  : Si no hay error: Tabla con los tokens de la consulta (FSQL_QUERY)
--           Si hay error   : Inserta en tabla FSQL_ERRORS el msg de Error
-- Devuelve:   0  Si no hay error
--           <>0  Si hay error
----------------------------------------------------------------------
FUNCTION fsql_lexico (consulta IN LONG) RETURN INTEGER IS
  MAXLONG_ID CONSTANT INTEGER    :=30;   -- Mxima long de un identificador (o token no cadena/texto)
  MAXLONG_CADTXT CONSTANT INTEGER:=2000; -- Mxima long de cadena/texto (2000=max long de una columna VARCHAR2)
  long     CONSTANT NUMBER :=length(consulta); -- Longitud de la consulta
  retor1   CONSTANT CHAR   :=CHR(13); -- Caracteres separadores
  retor2   CONSTANT CHAR   :=CHR(10);
  pos        INTEGER       :=1; -- Posicin del carcter que analizamos en la consulta
  lex_error  INTEGER       :=0; -- Error lexico 
  estado_act T_TRANSI.estado%TYPE :=0; -- Estado actual
  estado_sig T_TRANSI.sig_estado%TYPE; -- Estado siguiente en la tabla de transiciones
  cadena     VARCHAR2(32767):=''; -- Cadena donde se va almacenando el token
  car        CHAR;                -- Carcter que se est analizando
  MsgError   VARCHAR2(2000);      -- Mensaje del error (si lo hay)
  indi    FSQL_QUERY.indice%TYPE; -- Para insertar el pseudo-token que indica FIN
BEGIN
  -- Analizamos lxicamente TODA la consulta (no slo hasta encontrar el primer error)
  WHILE (pos<=long) LOOP
    car:=nls_upper(substr(consulta,pos,1)); -- Carcter i-simo en maysculas
    IF ascii(car)=9 THEN car:=' '; END IF;  -- Si es un tabulador se considera como un blanco

    BEGIN -- Calcular estado siguiente
      SELECT sig_estado INTO estado_sig FROM T_TRANSI
      WHERE (T_TRANSI.estado = estado_act) AND
            (T_TRANSI.caracter = ascii(car));
    EXCEPTION -- IF SQL%NOTFOUND THEN ...
      WHEN NO_DATA_FOUND THEN
           IF estado_act=23 OR estado_act=25 -- Si estamos en medio de una CADENA o TEXTO...
              OR estado_act=37               -- Comentario con '--'
              OR estado_act=39               -- Comentario con '\'
           THEN
              estado_sig:=estado_act;
           ELSIF estado_act=40 THEN -- Hay un * dentro de un comentario
              estado_sig:=39;
           ELSE estado_sig:=0;
           END IF;
    END;

    IF estado_sig<>0 THEN
       IF estado_act=23 OR estado_act=25 THEN   -- Si estamos en una CADENA o TEXTO...
--          dbms_output.put_line(':'||cadena||'-'||substr(consulta,pos,1));
          IF length(cadena)=MAXLONG_CADTXT THEN
             lex_error:=1;
             FSQL_AUX.inserta_error('ERROR LXICO en posicin '||pos||': '||
               'Cadena o texto demasiado larga.');
             cadena:=substr(consulta,pos,1);
          ELSE
             cadena:=cadena||substr(consulta,pos,1); -- concatenamos el carcter tal cual est
          END IF;
       ELSIF estado_act IN (37,39) THEN -- Comentario con '--'  con '\'
          NULL; -- No hacer nada
       ELSIF car<>' ' THEN -- Espacios entre <>, <=, >=, !=, ^= NO se unen
          IF length(cadena)=30 THEN
             lex_error:=1;
             FSQL_AUX.inserta_error('ERROR LXICO en posicin '||pos||': '||
               'Identificador tiene ms de 30 caracteres.');
             cadena:=car;
          ELSE
            cadena:=cadena||car;
          END IF;
       END IF;
       IF pos=long THEN -- FINAL de la consulta
--          dbms_output.put_line('FIN: '||cadena||'-'||estado_sig);
          lex_error:=inserta_token(cadena,estado_sig,pos);
       ELSE
          estado_act:=estado_sig;
       END IF;
    ELSIF estado_act<>0 THEN
--       dbms_output.put_line(cadena||'-'||estado_act);
       lex_error:=inserta_token(cadena,estado_act,pos-1);
       IF lex_error=0 THEN -- Si no ha habido errores...
          estado_act:=0;
          cadena:='';
          IF car <> ' ' And car <> retor1 And car <> retor2 THEN
             pos:=pos-1; -- Si un carcter, no separador, provoc el final del token,
          END IF;    -- resto 1, para no perder ese carcter (el sig. token empezar por l)
       ELSE --Si se ha producido un error, empezamos de nuevo (ese token es ignorado)
          estado_act:=0;
          cadena:='';
       END IF;
    ELSE -- estado_act=estado_sig=0
       IF car <> ' ' And car <> retor1 And car <> retor2 THEN
          lex_error:=1;
          FSQL_AUX.inserta_error('ERROR LXICO en posicin '||pos||': '
            ||'Smbolo '||car||' (ASCII '||ascii(car)
            ||') no reconocido por la gramtica y es ignorado.');
       END IF;
    END IF;

    pos:=pos+1;
  END LOOP;

  COMMIT;
  SELECT count(*) INTO lex_error FROM FSQL_ERRORS;
  RETURN lex_error;
END fsql_lexico;

-------------------------------------------------------------------------------------
-- ***** ***** ***** ***** ***** ANALIZ.  SINTCTICO ***** ***** ***** ***** ***** --
-------------------------------------------------------------------------------------

----------------------------------------------------------------------
-- Introduce en la pila todos los smbolos de la parte derecha de la produccin,
-- en orden inverso (de dcha a izda)
----------------------------------------------------------------------
PROCEDURE introducir_pila (PILA IN OUT TPILA, produccion IN INTEGER) IS
  CURSOR ParteDcha IS
         SELECT parte_der, terminal_der FROM PRODUCCIONES
         WHERE num_prod=produccion
         ORDER BY posicion DESC;

BEGIN
  FOR simbolo IN ParteDcha LOOP
--  dbms_output.put_line('Veamos: '||tope||','||simbolo.parte_der||','||simbolo.terminal_der);
    IF simbolo.parte_der <> 'VACIO' THEN -- Si la produccin no conduce a un nulo
      PILA.tope:=PILA.tope+1;
      PILA.simbolo (PILA.tope):=simbolo.parte_der;
      PILA.terminal(PILA.tope):=simbolo.terminal_der;
--      dbms_output.put_line('Insert: '||tope||','||simbolo.parte_der||','||simbolo.terminal_der);
    END IF;
  END LOOP;
END introducir_pila;

----------------------------------------------------------------------
-- Devuelve el smbolo del tope de la pila y si es Terminal ('T') o NT ('N')
----------------------------------------------------------------------
PROCEDURE leer_tope_pila (PILA IN OUT TPILA,
                          simbolo  OUT PRODUCCIONES.parte_der%TYPE,
                          terminal OUT PRODUCCIONES.terminal_der%TYPE) IS
BEGIN
  simbolo :=PILA.simbolo (PILA.tope);
  terminal:=PILA.terminal(PILA.tope);
  PILA.tope:=PILA.tope-1;
END leer_tope_pila;

----------------------------------------------------------------------
-- Devuelve el nombre del i-simo token (segn el orden de la consulta)
-- O sea, devuelve el campo FSQL_QUERY.nombre con FSQL_QUERY.indice=i
-- Si el nmero de tokens existentes es menor que i, devuelve FIN
----------------------------------------------------------------------
FUNCTION sig_token (i IN INTEGER) RETURN FSQL_QUERY.nombre%TYPE IS
  token FSQL_QUERY.nombre%TYPE;
  maxi  INTEGER;
BEGIN
  SELECT MAX(indice) INTO maxi FROM FSQL_QUERY;
  IF (i <= maxi) THEN
       SELECT nombre INTO token FROM FSQL_QUERY
         WHERE FSQL_QUERY.indice=i;
       RETURN token;
  ELSE RETURN FIN;
  END IF;
END sig_token;

----------------------------------------------------------------------
-- Cambia algunos smbolos terminales en el msg de error.
-- Msg de error del tipo: "Encontrado ...(1)... cuando esperaba ...(2)..."
-- El parmetro 'parte' indica la parte (1  2) donde va el smbolo que cambiamos
----------------------------------------------------------------------
FUNCTION cambia (simbT IN VARCHAR2,parte IN INTEGER) RETURN VARCHAR2 IS
BEGIN
  IF simbT=FIN THEN
     RETURN ' <fin>';
  ELSIF simbT='EQ'  THEN RETURN ' =';
  ELSIF simbT='NEQ' THEN RETURN ' <>';
  ELSIF simbT='GT'  THEN RETURN ' >';
  ELSIF simbT='LT'  THEN RETURN ' <';
  ELSIF simbT='LEQ' THEN RETURN ' <=';
  ELSIF simbT='GEQ' THEN RETURN ' >=';
  ELSIF simbT='ID'  THEN RETURN ' <identificador>';
  ELSIF simbT='NUMERO' THEN RETURN ' <nmero>';
  ELSIF simbT='CADENA' THEN RETURN ' <cadena>';
  ELSIF simbT='CONCAT' THEN RETURN ' ||';
  ELSIF simbT='TEXTO'  THEN RETURN ' <texto>';
  /* funciones: Se sustituyen todas por '' (nada), excepto ABS que se cambia por 'funcin' */
  ELSIF simbT='ABS'    THEN 
     IF parte=2 THEN RETURN ' <funcin>';
     END IF;
  ELSIF simbT='CDEG' OR simbT='CEIL' OR simbT='FLOOR' OR simbT='SIGN' OR
        simbT='SQRT' OR simbT='CHR' OR simbT='MOD' OR simbT='POWER' OR simbT='INITCAP' OR
        simbT='LOWER' OR simbT='LPAD' OR simbT='LTRIM' OR simbT='REPLACE' OR simbT='RPAD' OR
        simbT='RTRIM' OR simbT='SOUNDEX' OR simbT='SUBSTR' OR simbT='TRANSLATE' OR
        simbT='UPPER' OR simbT='ASCII' OR simbT='INSTR' OR simbT='LENGTH' OR
        simbT='NLSSORT' OR simbT='AVG' OR simbT='COUNT' OR simbT='MAX' OR simbT='MIN' OR
        simbT='STDDEV' OR simbT='SUM' OR simbT='VARIANCE' OR simbT='CHARTOROWID' OR
        simbT='CONVERT' OR simbT='HEXTORAW' OR simbT='RAWTOHEX' OR simbT='ROWIDTOCHAR' OR
        simbT='TO_CHAR' OR simbT='TO_DATE' OR simbT='TO_NUMBER' OR simbT='ADD_MONTHS' OR
        simbT='LAST_DAY' OR simbT='MONTHS_BETWEEN' OR simbT='NEW_TIME' OR
        simbT='NEXT_DAY' OR simbT='ROUND' OR simbT='TRUNC' OR simbT='DUMP' OR
        simbT='GREATEST' OR simbT='LEAST' OR simbT='NVL' OR simbT='USERENV' OR
        simbT='VSIZE' OR simbT='DECODE'
  THEN
     IF parte=2 THEN
        -- As, cuando hay una ristra de funciones (en la parte 2) slo se visualizar '<funcin>'
        RETURN '';
     END IF;
  END IF;
  RETURN ' '||simbT;
END cambia;

----------------------------------------------------------------------
-- ANALIZADOR SINTCTICO de los tokens de la tabla FSQL_QUERY, generados por el A. Lxico
-- Usa   : Tabla con los tokens ordenados (FSQL_QUERY)
--         Tabla con las producciones de la gramtica (PRODUCCIONES)
--         Tabla de transiciones segn la gramtica LL(1) (TABLA_SINTAX)
-- Genera: Si no hay error: Modifica tabla FSQL_QUERY con nueva informacin
--         Si hay error   : Inserta en tabla FSQL_ERRORS el msg de Error
-- Devuelve:   0  Si no hay error
--           <>0  Si hay error
----------------------------------------------------------------------
FUNCTION fsql_sintactico RETURN INTEGER IS
  entrada INTEGER :=0; -- Posicin del token de la entrada que se est analizando (desde 0)
  token         FSQL_QUERY.nombre%TYPE;         -- Nombre del token que se est analizando
  simbolo_tope  PRODUCCIONES.parte_der%TYPE;    -- Elemento Tope de la Pila
  terminal_tope PRODUCCIONES.terminal_der%TYPE; -- Si el tope es o no simbolo terminal
  sintax_error INTEGER :=0; -- Nmero del error sintctico (0 si no hay error).
  prod_a_aplicar TABLA_SINTAX.num_prod%TYPE;    -- Nmero de produccin a aplicar
  posicion       FSQL_QUERY.posicion%TYPE;      -- Posicin del error
  nombre_del_ID  FSQL_QUERY.atributo%TYPE:='';  -- Nombre del ID leido, caso de error
  MsgError       VARCHAR2(32767);               -- Msg de Error (si lo hay)
  -- Cursor para hallar los smbolos esperados (si se produce un error):
  CURSOR esperados (simbNT TABLA_SINTAX.simbolo_NT%TYPE) IS
     SELECT simbolo_T FROM TABLA_SINTAX
     WHERE  simbolo_NT=simbNT;
  PILA TPILA;
  funcion_activa BOOLEAN:=FALSE; -- Indica si estamos atravesando una llamada a funcin SQL
                                 -- As, marcamos los parntesis de dicha funcin, para que no confundan
  Into_Select_list BOOLEAN:=FALSE; -- Indica si estamos atravesando la select list.
                                   -- Se activa al leer un SELECT y se desactiva al leer un FROM
BEGIN
  -- Inicializar la PILA del Analizador:
  PILA.simbolo (1):=FIN;       -- Indica el FIN de las producciones lanzadas
  PILA.terminal(1):='N';
  PILA.tope:=1;

  -- Primer smbolo a analizar: El smbolo inicial
  simbolo_tope :=S_INICIAL; -- Smbolo inicial NT de la gramtica
  terminal_tope:='N';
  -- Primer token de la entrada
  token := sig_token(entrada);

  WHILE simbolo_tope<>FIN AND sintax_error=0 LOOP
--    dbms_output.put_line('--------------------');
    IF terminal_tope = 'T' THEN -- Si el simbolo del tope de la pila ES TERMINAL...
      IF simbolo_tope = token THEN -- Ok! token aceptado!
--         dbms_output.put_line(token||':'||prod_a_aplicar);

        -- Actualizamos el nombre del token aceptado con lo que sabemos por la produccin:
        -- Esto es til en el A. semntico.
        IF    token='SELECT' THEN  Into_Select_list:=TRUE;
        ELSIF token='FROM'   THEN  Into_Select_list:=FALSE;
        ELSIF token='CDEG' AND NOT Into_Select_list THEN
           SELECT posicion INTO posicion FROM FSQL_QUERY WHERE indice=entrada;
           FSQL_AUX.inserta_error('ERROR SINTCTICO en posicin '||posicion
             ||': Llamada a funcin CDEG mal situada. Debe aparecer en la SELECT_LIST.');
        ELSIF token='ID' THEN
            -- Columnas:
           IF   prod_a_aplicar=60 THEN
              UPDATE FSQL_QUERY set nombre='column' WHERE indice=entrada;
           ELSIF prod_a_aplicar=65 THEN
              UPDATE FSQL_QUERY set nombre='t.column'WHERE indice=entrada;
              UPDATE FSQL_QUERY set nombre='t.'  WHERE indice=entrada-2;
           ELSIF prod_a_aplicar=68 THEN
              UPDATE FSQL_QUERY set nombre='t.'  WHERE indice=entrada-2;
           ELSIF prod_a_aplicar=71 THEN
              UPDATE FSQL_QUERY set nombre='s.t.column'WHERE indice=entrada;
              UPDATE FSQL_QUERY set nombre='t.'        WHERE indice=entrada-2;
              UPDATE FSQL_QUERY set nombre='s.t.'      WHERE indice=entrada-4;
           ELSIF prod_a_aplicar=29 THEN
              UPDATE FSQL_QUERY set nombre='c_alias' WHERE indice=entrada;
           -- Tablas:
           ELSIF prod_a_aplicar=180 THEN
              UPDATE FSQL_QUERY set nombre='tabla' WHERE indice=entrada;
           ELSIF prod_a_aplicar=182 THEN
              UPDATE FSQL_QUERY set nombre='s.tabla'  WHERE indice=entrada;
              UPDATE FSQL_QUERY set nombre='t_scheme' WHERE indice=entrada-2;
           ELSIF prod_a_aplicar=186 THEN
              UPDATE FSQL_QUERY set nombre='t_alias' WHERE indice=entrada;
           END IF;
        ELSIF token='CADENA' AND prod_a_aplicar=28 THEN
           UPDATE FSQL_QUERY set nombre='c_alias' WHERE indice=entrada;
        ELSIF token=',' AND prod_a_aplicar=23 THEN -- Es una coma en la select_list
           UPDATE FSQL_QUERY set nombre=', en sl' WHERE indice=entrada;
        ELSIF token IN ('EQ','NEQ','LEQ','GEQ','LT','GT') -- Un comparador
              AND prod_a_aplicar BETWEEN 260 AND 265 THEN 
           UPDATE FSQL_QUERY set nombre='comparador' WHERE indice=entrada;
        ELSIF token='NUMERO' AND prod_a_aplicar=221 THEN -- Nmero como umbral
           UPDATE FSQL_QUERY set nombre='NUMBRAL' WHERE indice=entrada;
        ELSIF token='*' THEN
           IF prod_a_aplicar=149 THEN -- * de un CDEG(*)
              UPDATE FSQL_QUERY set nombre='*column' WHERE indice=entrada;
           ELSIF prod_a_aplicar=66 THEN
              UPDATE FSQL_QUERY set nombre='t.*' WHERE indice=entrada;
              UPDATE FSQL_QUERY set nombre='t.'  WHERE indice=entrada-2;
           ELSIF prod_a_aplicar=72 THEN
              UPDATE FSQL_QUERY set nombre='s.t.*' WHERE indice=entrada;
              UPDATE FSQL_QUERY set nombre='t.'    WHERE indice=entrada-2;
              UPDATE FSQL_QUERY set nombre='s.t.'  WHERE indice=entrada-4;
           ELSIF prod_a_aplicar=20 THEN -- Caso: SELECT * FROM ... --> Todas las columnas
              UPDATE FSQL_QUERY set nombre='TODAS' WHERE indice=entrada;
           END IF;
        ELSIF token='%' THEN
           IF prod_a_aplicar=67 THEN
              UPDATE FSQL_QUERY set nombre='t.%' WHERE indice=entrada;
              UPDATE FSQL_QUERY set nombre='t.'  WHERE indice=entrada-2;
           ELSIF prod_a_aplicar=73 THEN
              UPDATE FSQL_QUERY set nombre='s.t.%'WHERE indice=entrada;
              UPDATE FSQL_QUERY set nombre='t.'   WHERE indice=entrada-2;
              UPDATE FSQL_QUERY set nombre='s.t.' WHERE indice=entrada-4;
           ELSIF prod_a_aplicar=21 THEN
              -- Caso: SELECT % FROM ... --> Todas las columnas y los CDEG que tengan sentido
              UPDATE FSQL_QUERY set nombre='TODO' WHERE indice=entrada;
           END IF;

        ELSIF token='(' AND funcion_activa THEN
           UPDATE FSQL_QUERY set nombre='( function' WHERE indice=entrada;
        ELSIF token=')' AND funcion_activa THEN
           UPDATE FSQL_QUERY set nombre=') function' WHERE indice=entrada;
           funcion_activa:=FALSE;
        ELSIF prod_a_aplicar BETWEEN 90 AND 148 THEN
           funcion_activa:=TRUE;
        END IF;

         -- Ok: Leemos el siguiente smbolo de la pila y el siguiente token de la entrada
         leer_tope_pila (PILA, simbolo_tope, terminal_tope);
         entrada := entrada + 1;
         token := sig_token(entrada);
--         dbms_output.put_line('Nuevo token: '||token);
--         dbms_output.put_line('Nuevo tope : '||simbolo_tope);
      ELSE
        -- Error: Se espera simbolo_tope y en la entrada se encuentra token
        -- Para aceptar nmeros con signo negativo, donde se espera el token NUMERO
        -- se intercepta el caso en el que se encuentre un '-', se espere un NUMERO
        -- y el siguiente token al '-' sea un NUMERO:
        IF simbolo_tope='NUMERO' AND token='-' AND sig_token(entrada+1)='NUMERO' THEN
           -- Eliminamos el '-' y desplazamos todos los siguientes tokens
           DELETE FSQL_QUERY WHERE indice=entrada;
           UPDATE FSQL_QUERY set indice=indice-1 WHERE indice>entrada;
           -- Aadimos el '-' delante del nmero:
           UPDATE FSQL_QUERY set atributo='-'||atributo WHERE indice=entrada;
           IF prod_a_aplicar=221 THEN -- Si es un nmero como umbral, lo indicamos
              UPDATE FSQL_QUERY set nombre='NUMBRAL' WHERE indice=entrada;
           END IF;
           -- Ok: Leemos el siguiente smbolo de la pila y el siguiente token de la entrada
           leer_tope_pila (PILA, simbolo_tope, terminal_tope);
           entrada := entrada + 1;
           token := sig_token(entrada);
        ELSE
          sintax_error:=1;
        END IF;
      END IF;
    ELSE -- terminal_tope = 'N'
      BEGIN
        SELECT num_prod INTO prod_a_aplicar FROM TABLA_SINTAX WHERE
               (TABLA_SINTAX.simbolo_NT = simbolo_tope) AND
               (TABLA_SINTAX.simbolo_T  = token);
      EXCEPTION
        WHEN NO_DATA_FOUND THEN
          prod_a_aplicar:=0;
      END;

--      dbms_output.put_line(simbolo_tope||','||token||'->'||prod_a_aplicar);
      IF prod_a_aplicar <> 0 THEN
         -- Si una comparacin difusa no tiene umbral (se supone 1 por defecto),
         -- lo indicamos en el token donde debera estar el umbral. til en el A. semntico.
         IF prod_a_aplicar=223 AND token<>FIN THEN
            UPDATE FSQL_QUERY set nombre='sin umbral' WHERE indice=entrada;
         END IF;
         -- Aplicamos la produccion, metiendo su parte derecha en la pila:
         introducir_pila(PILA, prod_a_aplicar);
         -- Leemos el nuevo smbolo del tope de la pila:
         leer_tope_pila (PILA, simbolo_tope, terminal_tope);
      ELSE
         -- Error: No existe transicin para ese token:
         -- Se encuentra ese token y se esperaba Iniciales(simbolo_tope)
         sintax_error:=2;
      END IF;
    END IF;
  END LOOP;

  IF sintax_error<>0 THEN
     IF token=FIN THEN
        token:='el FIN';
        entrada:=entrada-1;
     END IF;

     -- Hallar posicin del error:
     SELECT posicion INTO posicion FROM FSQL_QUERY WHERE indice=entrada;
     IF token='ID' THEN -- Hallar el nombre de ese identificador para el msg de error
        SELECT ' '||atributo INTO nombre_del_ID FROM FSQL_QUERY WHERE indice=entrada;
     END IF;

     IF sintax_error=1 THEN
        FSQL_AUX.inserta_error('ERROR SINTCTICO en posicin '||posicion||
          ': Encontrado'||cambia(token,1)||nombre_del_ID||
          ' cuando esperaba'||cambia(simbolo_tope,2)||'.');
     ELSIF sintax_error=2 THEN
        MsgError:='ERROR SINTCTICO en posicin '||posicion||
          ': Encontrado'||cambia(token,1)||nombre_del_ID||' cuando esperaba: ';
        FOR esperado IN esperados(simbolo_tope) LOOP
            MsgError:=MsgError||cambia(esperado.simbolo_T,2);
        END LOOP;
        FSQL_AUX.inserta_error(MsgError);
     END IF;

  ELSE -- simbolo_tope=FIN
     IF token = FIN THEN --SIN ERRORES SINTCTICOS
        COMMIT;
        RETURN 0;
     ELSE
        -- Hallar posicin del error:
        SELECT posicion INTO posicion FROM FSQL_QUERY WHERE indice=entrada;
        FSQL_AUX.inserta_error('ERROR SINTCTICO en posicin '||posicion||
          ': Esperaba el FIN cuando encontr: '||token||'.');
     END IF;
  END IF;

  -- Ha habido errores: Los contamos y los devolvemos
  COMMIT;
  SELECT count(*) INTO sintax_error FROM FSQL_ERRORS;
  RETURN sintax_error;
END fsql_sintactico;


-------------------------------------------------------------------------------
-- ***** ***** CONVERSOR: FSQL --> SQL  (Funcin FSQL2SQL) ***** ***** ***** --
-------------------------------------------------------------------------------

----------------------------------------------------------------------
-- Actualiza valores en la vista FSQL_OPTIONS y algunas estadsticas de FSQL_STATS
-- Si se produce desbordamiento, se pone el 'contador' de nuevo a 0
----------------------------------------------------------------------
PROCEDURE Actualiza_FSQL_OPT (
  Nerrors      IN INTEGER,
  TIME_inicial IN DATE) IS
  TIME_final DATE;
  mini INTEGER;
  minf INTEGER;
  segi INTEGER;
  segf INTEGER;
  segundos INTEGER;
  SESSION_anterior NUMBER(38);
BEGIN
  -- Actualizar estadsticas sobre las sesiones:
  SELECT NUM INTO SESSION_anterior FROM FSQL_STATS WHERE EVENTO='ULTIMO_SESSIONID';
  IF SESSION_anterior<>USERENV('SESSIONID') THEN
     -- Actualizar ltimo nmero de sesin que accedi al Servidor FSQL
     UPDATE FSQL_STATS SET NUM=USERENV('SESSIONID') WHERE EVENTO='ULTIMO_SESSIONID';
     -- Actualizar nmero de cambios de sesin
     BEGIN
       UPDATE FSQL_STATS SET NUM=NUM+1 WHERE EVENTO='CAMBIOS_DE_SESSIONID';
     EXCEPTION WHEN OTHERS THEN
       UPDATE FSQL_STATS SET NUM=0     WHERE EVENTO='CAMBIOS_DE_SESSIONID';
     END;
  END IF;

  -- Calcular TIEMPO de traduccin:
  SELECT SYSDATE INTO TIME_final FROM DUAL;
  mini:=to_number(to_char(TIME_inicial,'MM'));
  segi:=to_number(to_char(TIME_inicial,'SS'));
  minf:=to_number(to_char(TIME_final,'MM'));
  segf:=to_number(to_char(TIME_final,'SS'));
  IF minf>=mini THEN
     segundos:=(minf*60 + segf) - (mini*60 + segi);
  ELSE
     segundos:=(minf*60 + segf + 3600) - (mini*60 + segi);
  END IF;
  IF segundos<0 THEN segundos:=1; END IF; -- Por si ocurre algn error inesperado!
--dbms_output.put_line(mini||':'||segi);
--dbms_output.put_line(minf||':'||segf||' --> '||segundos);

  -- Actualizar totales de nmero de accesos, sus tiempos y sus errores:
  BEGIN
    UPDATE FSQL_STATS SET NUM=NUM+segundos WHERE EVENTO='TIEMPO_TOTAL_ACCESOS';
  EXCEPTION WHEN OTHERS THEN
    UPDATE FSQL_STATS SET NUM=0            WHERE EVENTO='TIEMPO_TOTAL_ACCESOS';
  END;
  BEGIN
    UPDATE FSQL_STATS SET NUM=NUM+1 WHERE EVENTO='ACCESOS_PARA_TRADUCCIONES';
  EXCEPTION WHEN OTHERS THEN
    UPDATE FSQL_STATS SET NUM=0     WHERE EVENTO='ACCESOS_PARA_TRADUCCIONES';
  END;
  IF Nerrors=0 THEN
     BEGIN
       UPDATE FSQL_STATS SET NUM=NUM+segundos WHERE EVENTO='TIEMPO_ACCESOS_SIN_ERRORES';
     EXCEPTION WHEN OTHERS THEN
       UPDATE FSQL_STATS SET NUM=0            WHERE EVENTO='TIEMPO_ACCESOS_SIN_ERRORES';
     END;
     BEGIN
       UPDATE FSQL_STATS SET NUM=NUM+1        WHERE EVENTO='ACCESOS_SIN_ERRORES';
     EXCEPTION WHEN OTHERS THEN
       UPDATE FSQL_STATS SET NUM=0            WHERE EVENTO='ACCESOS_SIN_ERRORES';
     END;
  ELSE
     BEGIN
       UPDATE FSQL_STATS SET NUM=NUM+1        WHERE EVENTO='TOTAL_ERRORES_COMETIDOS';
     EXCEPTION WHEN OTHERS THEN
       UPDATE FSQL_STATS SET NUM=0            WHERE EVENTO='TOTAL_ERRORES_COMETIDOS';
     END;
  END IF;

  -- Actualizar Tiempo de acceso de la ultima consulta para el usuario actual:
  UPDATE FSQL_STATS SET NUM=segundos WHERE EVENTO='TIEMPO_ULTIMA_TRADUCCION';
  BEGIN
    INSERT INTO FSQL_OPTIONS VALUES (USER,'TIEMPO_TRADUCCION',
      TO_CHAR(segundos)||' Seg.');
  EXCEPTION WHEN OTHERS THEN
    UPDATE FSQL_OPTIONS SET VALOR=TO_CHAR(segundos)||' Seg.'
      WHERE OWNER=USER AND OPCION='TIEMPO_TRADUCCION';
  END;

  -- Actualizar nmero de errores del usuario actual:
  UPDATE FSQL_STATS SET NUM=Nerrors WHERE EVENTO='ERRORES_ULTIMA_TRADUCCION';
  BEGIN
    INSERT INTO FSQL_OPTIONS VALUES (USER,'NUM_ERRORES',TO_CHAR(Nerrors));
  EXCEPTION WHEN OTHERS THEN
    UPDATE FSQL_OPTIONS SET VALOR=TO_CHAR(Nerrors)
      WHERE OWNER=USER AND OPCION='NUM_ERRORES';
  END;

  COMMIT;
END Actualiza_FSQL_OPT;

----------------------------------------------------------------------
-- Funcin que traduce una consulta en FSQL a SQL.
-- Devuelve el nmero de errores cometidos: Lxicos, Sintcticos y Semnticos.
-- Al final de cada anlisis se hace un COMMIT, por lo que aqu no es necesario.
-- Si el nmero de errores es 0, la consulta SQL se obtiene concatenando los valores del
-- campo 'atributo' de las filas de la tabla FSQL_QUERY ordenadas por el campo 'indice'.
----------------------------------------------------------------------
FUNCTION FSQL2SQL (consulta IN LONG) RETURN INTEGER IS
  Nerrors INTEGER; -- Nmero de errores encontrados
  TIME_inicial DATE;
BEGIN
  -- Registramos el momento de comienzo:
  SELECT SYSDATE INTO TIME_inicial FROM DUAL;
  -- Calcular el nmero de sessin
  SELECT USERENV('SESSIONID') INTO AUDSID FROM DUAL;
  -- Actualizar la fecha de ltimo uso a la actual:
  UPDATE FSQL_ALL_INFO SET VALOR=to_char(SYSDATE,'DD-MM-YYYY, HH24:MI:SS')
    WHERE OWNER='FSQL SERVER' AND OPCION='FECHA_ULTIMO_USO';

  -- Borrn y cuenta nueva: Borramos los datos que hubiera antes de otra consulta:
  DELETE FROM FSQL_QUERY;
  DELETE FROM FSQL_ERRORS;

  IF consulta='' or consulta IS NULL THEN
     FSQL_AUX.inserta_error('ERROR: Consulta vaca.');
     RETURN 1;
  END IF;

  -- Si el primer carcter es ! no se traduce nada, pero ejecuta el A. lxico:
  IF substr(consulta,1,1)='!' THEN
     -- Se le quita el ! y se hace SLO el anlisis lxico
     RETURN fsql_pkg.fsql_lexico(substr(consulta,2,length(consulta)-1));
  END IF;

  Nerrors:=fsql_pkg.fsql_lexico(consulta);
  Nerrors :=fsql_pkg.fsql_sintactico; -- Incluye tambin los lxicos
  IF Nerrors<>0 THEN
     -- Actualizar valores en la vista FSQL_OPTIONS
     Actualiza_FSQL_OPT (Nerrors,TIME_inicial);
     RETURN Nerrors;
  ELSE
     -- No hacemos el Anlisis SEMANTICO si la consulta no esta correcta lxica
     -- y sintcticamente: Para no liar al A. Semntico y que no de errores innecesarios.
     Nerrors :=FSQL_SEMANTIC.semantico;
     Actualiza_FSQL_OPT (Nerrors,TIME_inicial);
     RETURN Nerrors;
  END IF;
END FSQL2SQL;

----------------------------------------------------------------------
-- Funcin que concatena el resultado SQL de una sentencia FSQL, devolviendo
-- la sentencia resultante en una cadena.
-- Posible problema: Que la consulta sea muy grande y exceda del tamao mximo para
-- cadenas de Oracle.
-- Tamaos mximos, por tipos (en Oracle7): CHAR (32767), LONG (32760) y VARCHAR2 (32767).
-- Esta funcin se incluye para:
--     a) Simplificar posibles llamadas al servidor si el resultado tiene menos
--        de 32767 caracteres, ya que realiza la concatenacin automticamente.
--        Salvo en sentencias muy grandes, es difcil sobrepasar ese lmite.
--     b) En futuras versiones de Oracle pueden incluirse tipos de datos con
--        mayores lmites.
-- Puede usarse tras FSQL2SQL, si no hubo errores.
----------------------------------------------------------------------
FUNCTION FSQL_concat RETURN VARCHAR2 IS
  anterior_sin_esp BOOLEAN;
  sentenciaSQL     VARCHAR2(32767);
  longitud_valor   NUMBER(5);
  longitud_total   NUMBER(5);
  ch               CHAR;
BEGIN
  -- No meter espacios ni antes ni despus de determinados caracteres
  anterior_sin_esp := FALSE;
  sentenciaSQL:='';
  longitud_total:=0;

  FOR valor IN (SELECT ATRIBUTO FROM FSQL_QUERY
                WHERE ATRIBUTO IS NOT NULL AND SESSIONID=USERENV('SESSIONID')
                ORDER BY INDICE) LOOP
     ch:= SUBSTR(valor.ATRIBUTO,1,1); -- Primer carcter de valor
     longitud_valor:=LENGTH(valor.ATRIBUTO);
     longitud_total:=longitud_total + longitud_valor;
     IF longitud_total>32767 THEN RETURN NULL;
     END IF;
     IF anterior_sin_esp OR
       (longitud_valor = 1 AND
       (ch = ',' OR ch = '.' OR ch = '=' OR ch = ';' OR ch = '-' OR ch = '+' OR
        ch = '(' OR ch = ')' OR ch = '/' OR ch = '*' OR ch = '>' OR ch = '<')) THEN
        sentenciaSQL := sentenciaSQL || valor.ATRIBUTO;
        anterior_sin_esp := NOT anterior_sin_esp;
     ELSE
        sentenciaSQL := sentenciaSQL || ' ' || valor.ATRIBUTO;
     END IF;
  END LOOP;

  RETURN sentenciaSQL;
END FSQL_concat;

----------------------------------------------------------------------
-- Borra las tuplas de FSQL_QUERY y FSQL_ERRORS utilizadas por la sessin actual.
-- Este procedimiento se DEBE usar cuando no se vayan a efectuar ms
-- consultas FSQL en la sesin actual.
-- NO se debe usar en cada nueva consulta FSQL, slo al final.
-- Adems, tambin se eliminan las tuplas de sesiones que no existan ya: Si existen
-- tuplas de este tipo es seal de que el Cliente de esa sesin olvid ejecutar este
-- procedimiento antes de salir de su sesin. Con esto conseguimos evitar que las
-- tablas FSQL_* crezcan demasiado (lo cual slo ocurrir si nadie ejecuta FSQL_FIN)
----------------------------------------------------------------------
PROCEDURE FSQL_FIN IS
BEGIN
  -- Borramos tuplas de la sesin actual
  DELETE FROM FSQL_QUERY;  -- WHERE SESSIONID=USERENV('SESSIONID');
  DELETE FROM FSQL_ERRORS; -- WHERE SESSIONID=USERENV('SESSIONID');
  -- Borramos tuplas de sesiones inexistentes (las existentes estn en V$SESSION)
  DELETE FROM FSQL_ALL_QUERIES WHERE SESSIONID NOT IN (SELECT AUDSID FROM V$SESSION);
  DELETE FROM FSQL_ALL_ERRORS  WHERE SESSIONID NOT IN (SELECT AUDSID FROM V$SESSION);
  -- Actualizar la fecha de ltimo uso de esta funcin:
  UPDATE FSQL_ALL_INFO SET VALOR=to_char(SYSDATE,'DD-MM-YYYY, HH24:MI:SS')
    WHERE OWNER='FSQL SERVER' AND OPCION='FECHA_ULTIMO_FIN';
  -- Actualizar nmero de ejecuciones de esta funcin
  BEGIN
    UPDATE FSQL_STATS SET NUM=NUM+1 WHERE EVENTO='EJECUCIONES_FSQL_FIN';
  EXCEPTION WHEN OTHERS THEN
    UPDATE FSQL_STATS SET NUM=0     WHERE EVENTO='EJECUCIONES_FSQL_FIN';
  END;
  COMMIT;
END FSQL_FIN;

----------------------------------------------------------------------
-- Dado un esquema.tabla (sch.tab), calcula y devuelve:
--	* Nmero de OBJ de la tabla (0 si no existe),
-- Este procedimiento puede ser usado por los clientes FSQL para efectuar
-- distintas operaciones con la FMB: Ver etiquetas de una columna...
----------------------------------------------------------------------
FUNCTION FSQL_OBJ (sch IN VARCHAR2, tab IN VARCHAR2)
  RETURN ACCESSIBLE_TABLES.OBJ#%TYPE
IS
  OBJ_ID ACCESSIBLE_TABLES.OBJ#%TYPE;
BEGIN
  BEGIN
    SELECT OBJ# into OBJ_ID FROM ACCESSIBLE_TABLES
      WHERE OWNER=sch AND TABLE_NAME=tab;
  EXCEPTION
    WHEN NO_DATA_FOUND THEN OBJ_ID:=0; -- No existe esa tabla    
  END;
  RETURN OBJ_ID;
END FSQL_OBJ;

----------------------------------------------------------------------
-- Dado un esquema.tabla.columna (sch.tab.col), calcula y devuelve:
--	* El COLUMN_ID de la columna (0 si no existe) y
-- Este procedimiento puede ser usado por los clientes FSQL para efectuar
-- distintas operaciones con la FMB: Ver etiquetas de una columna...
----------------------------------------------------------------------
FUNCTION FSQL_COL (sch IN VARCHAR2, tab IN VARCHAR2, col IN VARCHAR2)
  RETURN DBA_TAB_COLUMNS.COLUMN_ID%TYPE
IS
  COL_ID DBA_TAB_COLUMNS.COLUMN_ID%TYPE;
BEGIN
  BEGIN
    SELECT COLUMN_ID into COL_ID FROM DBA_TAB_COLUMNS
      WHERE OWNER=sch AND TABLE_NAME=tab AND COLUMN_NAME=col;
  EXCEPTION
    WHEN NO_DATA_FOUND THEN COL_ID:=0; -- No existe esa columna
  END;
  RETURN COL_ID;
END FSQL_COL;

----------------------------------------------------------------------
-- Dado un esquema.tabla.columna (sch.tab.col), calcula y devuelve:
--	* Tipo difuso que tiene en FCL (0 si no es difusa).
-- Este procedimiento puede ser usado por los clientes FSQL para efectuar
-- distintas operaciones con la FMB: Ver etiquetas de una columna...
----------------------------------------------------------------------
FUNCTION FSQL_FTYPE (
  OBJ_ID IN ACCESSIBLE_TABLES.OBJ#%TYPE,
  COL_ID IN DBA_TAB_COLUMNS.COLUMN_ID%TYPE)
  RETURN DBA_TAB_COLUMNS.COLUMN_ID%TYPE
IS
  F_TYP FUZZY_COL_LIST.F_TYPE%TYPE;
BEGIN
  BEGIN
    SELECT F_TYPE into F_TYP FROM FUZZY_COL_LIST
      WHERE OBJ#=OBJ_ID AND COL#=COL_ID;
  EXCEPTION
    WHEN NO_DATA_FOUND THEN F_TYP:=0; -- Esa columna no es difusa
  END;
  RETURN F_TYP;
END FSQL_FTYPE;

BEGIN
  -- Actualizar dato en FSQL_STATS
  UPDATE FSQL_STATS SET NUM=NUM+1 WHERE EVENTO='TOTAL_PAQUETE_FSQL_PKG';
EXCEPTION WHEN OTHERS THEN
  UPDATE FSQL_STATS SET NUM=0     WHERE EVENTO='TOTAL_PAQUETE_FSQL_PKG';
END FSQL_PKG;
/
